"use client";

import { useState, useEffect } from "react";
import Link from "next/link";
import { usePathname } from "next/navigation";
import AppLogo from "@/components/ui/AppLogo";
import Icon from "@/components/ui/AppIcon";

const navLinks = [
  { label: "Início", href: "/homepage" },
  { label: "Sobre", href: "/homepage#sobre" },
  { label: "Sistemas", href: "/systems" },
  { label: "Preços", href: "/homepage#precos" },
];

export default function Header() {
  const [scrolled, setScrolled] = useState(false);
  const [mobileOpen, setMobileOpen] = useState(false);
  const pathname = usePathname();

  useEffect(() => {
    const handleScroll = () => setScrolled(window.scrollY > 20);
    window.addEventListener("scroll", handleScroll, { passive: true });
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  useEffect(() => {
    setMobileOpen(false);
  }, [pathname]);

  const isActive = (href: string) => {
    if (href === "/homepage") return pathname === "/homepage";
    if (href === "/systems") return pathname === "/systems";
    return false;
  };

  return (
    <>
      <header
        className={`fixed top-0 left-0 right-0 z-50 transition-all duration-500 ${
          scrolled
            ? "bg-white/95 backdrop-blur-xl shadow-soft border-b border-primary-50"
            : "bg-transparent"
        }`}
      >
        <div className="max-w-7xl mx-auto px-6 lg:px-8">
          <div className="flex items-center justify-between h-20">
            {/* Logo */}
            <Link href="/homepage" className="flex items-center gap-3 group">
              <AppLogo
                size={64}
                text="ADVOPACK"
                iconName="CpuChipIcon"
                className="transition-transform duration-300 group-hover:scale-105"
              />
            </Link>

            {/* Desktop Nav */}
            <nav className="hidden md:flex items-center gap-1">
              {navLinks.map((link) => (
                <Link
                  key={link.href}
                  href={link.href}
                  className={`relative px-4 py-2 text-sm font-semibold rounded-lg transition-all duration-300 font-heading ${
                    isActive(link.href)
                      ? "text-primary bg-primary-50"
                      : scrolled
                      ? "text-gray-600 hover:text-primary hover:bg-primary-50" :"text-gray-700 hover:text-primary hover:bg-white/60"
                  }`}
                >
                  {link.label}
                  {isActive(link.href) && (
                    <span className="absolute bottom-0 left-1/2 -translate-x-1/2 w-4 h-0.5 bg-primary rounded-full" />
                  )}
                </Link>
              ))}
            </nav>

            {/* CTA */}
            <div className="hidden md:flex items-center gap-3">
              <Link
                href="/homepage#contato"
                className="btn-primary text-sm px-6 py-3"
              >
                <Icon name="ChatBubbleLeftRightIcon" size={16} />
                Solicitar Demo
              </Link>
            </div>

            {/* Mobile Toggle */}
            <button
              onClick={() => setMobileOpen(!mobileOpen)}
              className="md:hidden p-2 rounded-xl text-primary hover:bg-primary-50 transition-colors"
              aria-label="Menu"
            >
              <Icon name={mobileOpen ? "XMarkIcon" : "Bars3Icon"} size={24} />
            </button>
          </div>
        </div>
      </header>

      {/* Mobile Menu */}
      <div
        className={`fixed inset-0 z-40 md:hidden transition-all duration-300 ${
          mobileOpen ? "opacity-100 pointer-events-auto" : "opacity-0 pointer-events-none"
        }`}
      >
        <div
          className="absolute inset-0 bg-primary-dark/60 backdrop-blur-sm"
          onClick={() => setMobileOpen(false)}
        />
        <div
          className={`absolute top-0 right-0 h-full w-80 bg-white shadow-elevated transition-transform duration-400 ease-spring ${
            mobileOpen ? "translate-x-0" : "translate-x-full"
          }`}
        >
          <div className="flex items-center justify-between p-6 border-b border-primary-50">
            <AppLogo size={48} text="ADVOPACK" iconName="CpuChipIcon" />
            <button
              onClick={() => setMobileOpen(false)}
              className="p-2 rounded-xl text-gray-500 hover:bg-gray-100"
            >
              <Icon name="XMarkIcon" size={20} />
            </button>
          </div>
          <nav className="p-6 flex flex-col gap-2">
            {navLinks.map((link) => (
              <Link
                key={link.href}
                href={link.href}
                className={`px-4 py-3 rounded-xl text-sm font-semibold font-heading transition-all duration-200 ${
                  isActive(link.href)
                    ? "bg-primary text-white" :"text-gray-600 hover:bg-primary-50 hover:text-primary"
                }`}
              >
                {link.label}
              </Link>
            ))}
            <div className="mt-4 pt-4 border-t border-gray-100">
              <Link
                href="/homepage#contato"
                className="btn-primary w-full justify-center text-sm"
              >
                <Icon name="ChatBubbleLeftRightIcon" size={16} />
                Solicitar Demo
              </Link>
            </div>
          </nav>
        </div>
      </div>
    </>
  );
}