"use client";

import { useEffect, useRef } from "react";
import Icon from "@/components/ui/AppIcon";

interface DemoModalProps {
  isOpen: boolean;
  onClose: () => void;
  system: {
    name: string;
    demoUrl: string;
    segment: string;
  } | null;
}

export default function DemoModal({ isOpen, onClose, system }: DemoModalProps) {
  const overlayRef = useRef<HTMLDivElement>(null);
  const iframeRef = useRef<HTMLIFrameElement>(null);

  useEffect(() => {
    if (isOpen) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "";
    }
    return () => {
      document.body.style.overflow = "";
    };
  }, [isOpen]);

  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if (e.key === "Escape") onClose();
    };
    if (isOpen) {
      document.addEventListener("keydown", handleKeyDown);
    }
    return () => document.removeEventListener("keydown", handleKeyDown);
  }, [isOpen, onClose]);

  const handleOverlayClick = (e: React.MouseEvent<HTMLDivElement>) => {
    if (e.target === overlayRef.current) onClose();
  };

  const handleOpenNewTab = () => {
    if (system?.demoUrl) {
      window.open(system.demoUrl, "_blank", "noopener,noreferrer");
    }
  };

  if (!isOpen || !system) return null;

  return (
    <div
      ref={overlayRef}
      className="modal-overlay"
      onClick={handleOverlayClick}
      role="dialog"
      aria-modal="true"
      aria-label={`Demo: ${system.name}`}
    >
      <div className="modal-container">
        {/* Modal Header */}
        <div className="modal-header">
          <div className="flex items-center gap-3">
            <div className="w-8 h-8 rounded-lg bg-white/20 flex items-center justify-center">
              <Icon name="ComputerDesktopIcon" size={16} className="text-white" />
            </div>
            <div>
              <p className="text-white font-heading font-bold text-sm">{system.name}</p>
              <p className="text-white/60 text-xs">{system.segment} · Demo ao vivo</p>
            </div>
            <span className="ml-3 flex items-center gap-1.5 bg-white/10 border border-white/20 rounded-full px-3 py-1 text-xs text-white/80 font-medium">
              <span className="w-1.5 h-1.5 rounded-full bg-green-400 animate-pulse" />
              Ao Vivo
            </span>
          </div>

          <div className="flex items-center gap-2">
            {/* URL bar (decorative) */}
            <div className="hidden md:flex items-center gap-2 bg-white/10 rounded-lg px-3 py-1.5 max-w-xs">
              <Icon name="LockClosedIcon" size={12} className="text-green-400" />
              <span className="text-white/60 text-xs truncate font-mono">{system.demoUrl}</span>
            </div>

            {/* Open in new tab */}
            <button
              onClick={handleOpenNewTab}
              className="flex items-center gap-2 bg-white/15 hover:bg-white/25 border border-white/20 rounded-lg px-3 py-1.5 text-white text-xs font-semibold transition-all duration-200"
              title="Abrir em nova aba"
            >
              <Icon name="ArrowTopRightOnSquareIcon" size={14} />
              <span className="hidden sm:inline">Nova aba</span>
            </button>

            {/* Close */}
            <button
              onClick={onClose}
              className="w-8 h-8 rounded-lg bg-white/15 hover:bg-red-500/80 flex items-center justify-center text-white transition-all duration-200"
              title="Fechar"
              aria-label="Fechar modal"
            >
              <Icon name="XMarkIcon" size={16} />
            </button>
          </div>
        </div>

        {/* Loading state + iframe */}
        <div className="flex-1 relative bg-gray-50">
          {/* Loading skeleton */}
          <div className="absolute inset-0 flex items-center justify-center bg-gradient-to-br from-primary-50 to-white z-10 pointer-events-none" id="iframe-loading">
            <div className="text-center">
              <div className="w-16 h-16 rounded-2xl bg-primary/10 flex items-center justify-center mx-auto mb-4 animate-pulse">
                <Icon name="GlobeAltIcon" size={32} className="text-primary" />
              </div>
              <p className="text-primary font-heading font-semibold text-sm">Carregando demo...</p>
              <p className="text-gray-400 text-xs mt-1">Aguarde um momento</p>
            </div>
          </div>

          <iframe
            ref={iframeRef}
            src={system.demoUrl}
            className="modal-iframe"
            title={`Demo do sistema ${system.name}`}
            onLoad={() => {
              const loading = document.getElementById("iframe-loading");
              if (loading) loading.style.display = "none";
            }}
            sandbox="allow-same-origin allow-scripts allow-forms allow-popups"
          />
        </div>

        {/* Modal Footer */}
        <div className="bg-gray-50 border-t border-gray-100 px-6 py-3 flex items-center justify-between flex-shrink-0">
          <p className="text-xs text-gray-400">
            Esta é uma demonstração do sistema. Dados exibidos são fictícios.
          </p>
          <button
            onClick={onClose}
            className="text-xs font-semibold text-primary hover:text-accent transition-colors"
          >
            Fechar demo
          </button>
        </div>
      </div>
    </div>
  );
}